/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package printing;

import java.awt.print.PrinterJob;
import java.io.IOException;
import java.net.URL;

import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.standard.Copies;
import javax.print.attribute.standard.JobName;
import javax.print.attribute.standard.MediaSizeName;
import javax.print.attribute.standard.SheetCollate;

import com.inet.report.EngineRenderData;
import com.inet.viewer.PrinterJobProgress;
import com.inet.viewer.Progress;
import com.inet.viewer.RenderData;

/**
 * This example demonstrates how to print to a custom printer with the i-net Clear Reports viewer API without showing
 * the viewer. We set the page and job attributes and create a PrinterJobProgress. The report is then printed
 * asynchronously.
 */
public class CustomPrinter {

    /**
     * Print the report on a custom printer
     * @param reportUrl the URL of the report
     * @param printer a list of printers
     */
    static void print( String reportUrl, String... printer ) {

        // then initialize the render data connection.
        RenderData renderConnection = new EngineRenderData( "report=" + reportUrl );

        // you will most likely have a report server already, so you can use the URL: http://server:port/report.rpt
        // RenderData renderConnection = new URLRenderData( "http://server:port/?report=file:c:/report1.rpt" );

        // Construct the print request specification.
        // The print data is a Printable object.
        // The request additionally specifies a job name, 2 copies, and
        // portrait orientation of the media.
        PrintRequestAttributeSet printAttributes = new HashPrintRequestAttributeSet();
        // alternative with default attributes from report
        /* SwingReportView view = (SwingReportView)new SwingReportViewer().createReportView( renderConnection );
        PrintRequestAttributeSet printAttributes = view.getDefaultAttributeSet( 1, -1 );
        */

        // set some custom (optional) print properties
        printAttributes.add( new JobName( "My job", null ) );
        printAttributes.add( MediaSizeName.ISO_A4 );
        // aset.add(OrientationRequested.PORTRAIT);   // AUTO from report but you can override it
        printAttributes.add( new Copies( 2 ) );
        printAttributes.add( SheetCollate.COLLATED );

        // now print to all the specified printers
        for( int i = 0; i < printer.length; i++ ) {
            // locate a print service that can handle the request
            PrintService[] services = PrintServiceLookup.lookupPrintServices( null, null );

            // Search all printers to find the one that matches the provided name
            int whichPrinter = -1;
            for( int p = 0; p < services.length; p++ ) {
                if( services[p].getName().equals( printer[i] ) ) {
                    whichPrinter = p;
                    break;
                }
            }
            if( whichPrinter == -1 ) {
                System.out.println( "*** Printer " + printer[i] + " could not be found." );
                continue;
            }
            System.out.println( "\tselected printer: " + services[whichPrinter].getName() );
            try {
                PrinterJob pj = PrinterJob.getPrinterJob();
                pj.setPrintService( services[whichPrinter] );

                Progress progress = new PrinterJobProgress( null, pj, printAttributes, renderConnection );
                progress.startProgress();

                // We wait because we terminate the sample if the printing is finish.
                // In a read application there is no need to wait on the asynchronous print.
                progress.waitUntilFinished();

                System.out.println( " printer job finished." );
            } catch( Exception e ) {
                e.printStackTrace(); // handle the exception, for example show it to the user or write it to a log file
            }
        }
    }

    /**
     * Main method of the sample.
     * @param args no parameters used
     * @throws IOException in case of IO error (e.g. port already in use)
     */
    public static void main( String[] args ) throws IOException {
        URL url = CustomPrinter.class.getResource( "../sample.rpt" );
        if( url == null ) {
            System.out.println( "Report start.rpt not found." );
            return;
        }
        // The URL of a report file.
        String reportUrl = url.toString();

        /*
         * An array of printers we have.  We have obtained this list by
         * clicking on the print button of the Java viewer and then choose the
         * "select printer dialog".  -- You can also call the method
         * bean.printView() without any parameters to see the printer dialog.
         */
        String[] printer = { "Lexmark Optra E312 (MS)", "CutePDF Writer" };

        // print the rpt file
        print( reportUrl, printer );
    }
}
